#ifndef __CColour__
#define __CColour__

#include <Basics/CCountedObject.hpp>
using Exponent::Basics::CCountedObject;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Basics
		{
			/**
			 * @class CColour CColour.hpp
			 * @brief Representation of an RGB colour
			 *
			 * @date 20/09/2004
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CColour.hpp,v 1.4 2007/02/08 21:07:54 paul Exp $
			 */
			class CColour : public CCountedObject
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				const static CColour CCOLOUR_BLACK;			/**< Black */
				const static CColour CCOLOUR_WHITE;			/**< White */
				const static CColour CCOLOUR_RED;			/**< Red */
				const static CColour CCOLOUR_GREEN;			/**< Green */
				const static CColour CCOLOUR_BLUE;			/**< Blue */
				const static CColour CCOLOUR_YELLOW;		/**< Yellow */
				const static CColour CCOLOUR_ORANGE;		/**< Orange */
				const static CColour CCOLOUR_TURQUOISE;		/**< Turqoise */
				const static CColour CCOLOUR_PURPLE;		/**< Purple */
				const static CColour CCOLOUR_PINK;			/**< Pink */
				const static CColour CCOLOUR_LIGHT_GREY;	/**< Light grey */
				const static CColour CCOLOUR_DARK_GREY;		/**< Dark grey */

//	===========================================================================

				/**
				 * Construction
				 * @param red The red component
				 * @param green The green component
				 * @param blue The blue component
				 */
				CColour(const unsigned long red, const unsigned long green, const unsigned long blue);

				/**
				 * Construction
				 */
				CColour();

				/**
				 * Copy construction
				 * @param colour The colour to copy
				 */
				CColour(const CColour &colour);

				#ifdef WIN32
					/**
					 * Construction from a COLOURREF
					 * @param colour The colour to copy
					 */
					CColour(COLORREF &colour);
				#else
					/**
					 * Coinstruction from an RGBColour
					 * @param colour The colour to copy
					 */
					CColour(const RGBColor &colour);
				#endif

				/**
				 * Destruction
				 */
				virtual ~CColour();

//	===========================================================================

				/**
				 * Assignment operator
				 * @param colour The colour to copy
				 * @retval CColour& A reference to this
				 */
				CColour &operator = (const CColour &colour);

				/**
				 * Add two colours
				 * @param colour The colour to merge with
				 * @retval CColour& A reference to this
				 */
				CColour &operator += (const CColour &colour);

				/**
				 * Equality operator
				 * @param colour the colour to compare to
				 * @retval bool True if the colours are the same, false otherwise
				 */
				bool operator == (const CColour &colour);

//	===========================================================================

				/**
				 * Set the red
				 * @param red The red component
				 */
				void setRed(const unsigned long red) { m_red = red; }

				/**
				 * Set the green
				 * @param green The green component
				 */
				void setGreen(const unsigned long green) { m_green = green; }

				/**
				 * Set the blue
				 * @param blue The blue component
				 */
				void setBlue(const unsigned long blue) { m_blue = blue; }

				/**
				 * Set the colour
				 * @param red The red component
				 * @param green The green component
				 * @param blue The blue component
				 */
				virtual void setColour(const unsigned long red, const unsigned long green, const unsigned long blue);

//	===========================================================================

				/**
				 * Get the red
				 * @retval long The red component
				 */
				unsigned long getRed() const { return m_red; }

				/**
				 * Get the green
				 * @retval long The green component
				 */
				unsigned long getGreen() const { return m_green; }

				/**
				 * Get the blue
				 * @retval long The blue component
				 */
				unsigned long getBlue() const { return m_blue; }

//	===========================================================================

				#ifdef WIN32
					/**
					 * Get this as a colour ref
					 * @retval COLORREF The window colourref
					 */
					COLORREF getAsColourRef() const;

					/**
					 * SEt from a colour
					 * @param colour The colour to copy
					 */
					void setFromColourRef(const COLORREF &colour);
				#else
					/**
					 * Get colour as an RGBColour - you own returned value
					 * @retval RGBColor* This as an RGBColor
					 */
					RGBColor *getAsRGBColour() const;

					/**
					 * Set from an REGBcolour
					 * @param colour The colour to copy
					 */
					void setFromRGBColour(const RGBColor &colour);
				#endif

//	===========================================================================

				/**
				 * Get a description of the object
				 * @param string On return is filled with the description
				 * @param size The size of the stirng
				 */
				virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

			protected:

//	===========================================================================
			
				#ifndef WIN32
					/**
					 * Convert an exponent range to an apple range
					 * @param value The exponent ranged value
					 * @retval unsigned short The apple ranged value
					 */
					static unsigned short toAppleValue(const long value);
					
					/**
					 * Convert from an apple value to exponent
					 * @param appleValue The apple ranged value
					 * @retval long The exponent ranged value
					 */
					static long fromAppleValue(const unsigned short appleValue);
				#endif

//	===========================================================================

				unsigned long m_red;			/**< Red component */
				unsigned long m_green;			/**< Green component */
				unsigned long m_blue;			/**< Blue component */

			};
		}
	}
}
#endif	// End of CColour.hpp